import React, { useEffect, useState } from "react";
import api from "../../api";
import { Layout } from "../../components/Layout";
import { StatCard } from "../../components/StatCard";
import { DataTable } from "../../components/DataTable";
import {
  FaCheck,
  FaTimes,
  FaUsers,
  FaClock,
  FaCheckCircle,
  FaTimesCircle,
} from "react-icons/fa";

export const Dashboard = () => {
  const [leaves, setLeaves] = useState([]);

  const fetchLeaves = async () => {
    try {
      const res = await api.get("/leave/all");
      setLeaves(res.data);
    } catch (err) {
      console.log(err);
    }
  };

  useEffect(() => {
    fetchLeaves();
  }, []);

  const updateStatus = async (id, status) => {
    try {
      await api.put(`/leave/${id}`, { status });
      fetchLeaves();
    } catch (err) {
      console.log(err);
    }
  };

  const columns = [
    {
      header: "Employee",
      accessor: "user",
      render: (row) => row.user?.name || "Unknown",
    },
    {
      header: "From",
      accessor: "from",
      render: (row) => new Date(row.from).toLocaleDateString(),
    },
    {
      header: "To",
      accessor: "to",
      render: (row) => new Date(row.to).toLocaleDateString(),
    },
    { header: "Reason", accessor: "reason" },
    {
      header: "Status",
      accessor: "status",
      render: (row) => (
        <span
          className={`px-3 py-1 rounded-full text-xs font-semibold ${
            row.status === "approved"
              ? "bg-green-100 text-green-700"
              : row.status === "rejected"
                ? "bg-red-100 text-red-700"
                : "bg-yellow-100 text-yellow-700"
          }`}
        >
          {row.status.charAt(0).toUpperCase() + row.status.slice(1)}
        </span>
      ),
    },
  ];

  const actions = (row) =>
    row.status === "pending" && (
      <>
        <button
          onClick={() => updateStatus(row._id, "approved")}
          className="p-2 bg-green-100 text-green-600 rounded-full hover:bg-green-200 transition-colors"
          title="Approve"
        >
          <FaCheck size={14} />
        </button>
        <button
          onClick={() => updateStatus(row._id, "rejected")}
          className="p-2 bg-red-100 text-red-600 rounded-full hover:bg-red-200 transition-colors"
          title="Reject"
        >
          <FaTimes size={14} />
        </button>
      </>
    );

  // Stats calculation
  const totalLeaves = leaves.length;
  const pendingLeaves = leaves.filter((l) => l.status === "pending").length;
  const approvedLeaves = leaves.filter((l) => l.status === "approved").length;
  const rejectedLeaves = leaves.filter((l) => l.status === "rejected").length;

  return (
    <Layout>
      <div className="space-y-6">
        <h2 className="text-2xl font-bold text-gray-800">Admin Dashboard</h2>

        <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
          <StatCard
            title="Total Requests"
            value={totalLeaves}
            icon={<FaUsers />}
            color="bg-blue-500"
          />
          <StatCard
            title="Pending"
            value={pendingLeaves}
            icon={<FaClock />}
            color="bg-yellow-500"
          />
          <StatCard
            title="Approved"
            value={approvedLeaves}
            icon={<FaCheckCircle />}
            color="bg-green-500"
          />
          <StatCard
            title="Rejected"
            value={rejectedLeaves}
            icon={<FaTimesCircle />}
            color="bg-red-500"
          />
        </div>

        <div className="bg-white rounded-lg shadow-md p-6">
          <h3 className="text-lg font-semibold text-gray-700 mb-4">
            Leave Requests
          </h3>
          <DataTable columns={columns} data={leaves} actions={actions} />
        </div>
      </div>
    </Layout>
  );
};
